// license:BSD-3-Clause
// copyright-holders:Nigel Barnes
/**********************************************************************

    Beeb Speech Synthesiser - Watford Electronics

    TODO:
    - verify clock and low/high intonation
    - add reset button

**********************************************************************/

#include "emu.h"
#include "beebspch.h"

#include "sound/sp0256.h"

#include "speaker.h"


namespace {

class bbc_beebspch_device : public device_t, public device_bbc_userport_interface
{
public:
	bbc_beebspch_device(const machine_config &mconfig, const char *tag, device_t *owner, uint32_t clock)
		: device_t(mconfig, BBC_BEEBSPCH, tag, owner, clock)
		, device_bbc_userport_interface(mconfig, *this)
		, m_nsp(*this, "sp0256")
	{
	}

	void cb1_w(int state);
	void cb2_w(int state);

protected:
	// device_t overrides
	virtual void device_start() override ATTR_COLD { }

	// optional information overrides
	virtual void device_add_mconfig(machine_config &config) override ATTR_COLD;
	virtual const tiny_rom_entry *device_rom_region() const override ATTR_COLD;

	virtual void pb_w(uint8_t data) override;

private:
	required_device<sp0256_device> m_nsp;
};


//-------------------------------------------------
//  rom_region - device-specific ROM region
//-------------------------------------------------

ROM_START(beebspch)
	ROM_REGION(0x4000, "exp_rom", 0)
	ROM_LOAD("watford_speech.rom", 0x0000, 0x2000, CRC(731642a8) SHA1(1bd31345af6043f394bc9d8e65180c93b2356905))
	ROM_RELOAD(0x2000, 0x2000)

	ROM_REGION(0x10000, "sp0256", 0)
	ROM_LOAD("sp0256a-al2.bin", 0x1000, 0x0800, CRC(b504ac15) SHA1(e60fcb5fa16ff3f3b69d36c7a6e955744d3feafc))
ROM_END

const tiny_rom_entry *bbc_beebspch_device::device_rom_region() const
{
	return ROM_NAME(beebspch);
}


//-------------------------------------------------
//  device_add_mconfig - add device configuration
//-------------------------------------------------

void bbc_beebspch_device::device_add_mconfig(machine_config &config)
{
	SPEAKER(config, "mono").front_center();
	SP0256(config, m_nsp, 3120000); // TODO: generated by a LS269
	m_nsp->data_request_callback().set(FUNC(bbc_beebspch_device::cb1_w));
	m_nsp->standby_callback().set(FUNC(bbc_beebspch_device::cb2_w));
	m_nsp->add_route(ALL_OUTPUTS, "mono", 1.0);
}


//**************************************************************************
//  IMPLEMENTATION
//**************************************************************************

void bbc_beebspch_device::pb_w(uint8_t data)
{
	switch (data & 0xc0)
	{
	case 0x40:
		// intonation high
		m_nsp->set_clock(3800000); // TODO: the exact frequency is unknown
		break;
	case 0x80:
		// intonation low
		m_nsp->set_clock(3500000); // CK / 4 ??
		break;
	}

	// allophone
	m_nsp->ald_w(data & 0x3f);
}

void bbc_beebspch_device::cb1_w(int state)
{
	m_slot->cb1_w(state);
}

void bbc_beebspch_device::cb2_w(int state)
{
	m_slot->cb2_w(state);
}

} // anonymous namespace


DEFINE_DEVICE_TYPE_PRIVATE(BBC_BEEBSPCH, device_bbc_userport_interface, bbc_beebspch_device, "bbc_beebspch", "Beeb Speech Synthesiser")
