# Copyright (c) 2015 SUSE Linux GmbH.  All rights reserved.
#
# This file is part of kiwi.
#
# kiwi is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# kiwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with kiwi.  If not, see <http://www.gnu.org/licenses/>
#
import logging
from typing import (
    List, Optional
)

# project
from kiwi.command import Command
from kiwi.partitioner.base import PartitionerBase

from kiwi.exceptions import (
    KiwiPartitionerGptFlagError
)

log = logging.getLogger('kiwi')


class PartitionerGpt(PartitionerBase):
    """
    **Implements GPT partition setup**
    """
    def post_init(self) -> None:
        """
        Post initialization method

        Setup gdisk partition type/flag map
        """
        self.flag_map = {
            'f.active': None,
            't.csm': 'EF02',
            't.linux': '8300',
            't.swap': '8200',
            't.lvm': '8E00',
            't.raid': 'FD00',
            't.efi': 'EF00',
            't.prep': '4100'
        }
        self.partition_map: dict[int, int] = {}

    def create(
        self, name: str, mbsize: int, type_name: str, flags: List[str] = None,
        partition_id: Optional[int] = None
    ) -> None:
        """
        Create GPT partition

        :param string name: partition name
        :param int mbsize: partition size
        :param string type_name: partition type
        :param list flags: additional flags
        :param int partition_id:
            If provided, use this exact partition ID
            instead of auto-incrementing
        """
        self.partition_id = \
            partition_id if partition_id else self.partition_id + 1
        self.partition_count += 1
        self.partition_map[self.partition_count] = self.partition_id
        if mbsize == 'all_free':
            partition_end = '0'
        else:
            partition_end = '+' + format(mbsize) + 'M'
        if self.partition_count > 1 or not self.start_sector:
            # A start  sector value of 0 specifies the default value
            # defined in sgdisk
            self.start_sector = 0
        Command.run(
            [
                'sgdisk', '-n', ':'.join(
                    [
                        format(self.partition_id),
                        format(self.start_sector),
                        partition_end
                    ]
                ), '-c', ':'.join([format(self.partition_id), name]),
                self.disk_device
            ]
        )
        self.set_flag(self.partition_id, type_name)
        if flags:
            for flag_name in flags:
                self.set_flag(self.partition_id, flag_name)

    def set_flag(self, partition_id: int, flag_name: str) -> None:
        """
        Set GPT partition flag

        :param int partition_id: partition number
        :param string flag_name: name from flag map
        """
        if flag_name not in self.flag_map:
            raise KiwiPartitionerGptFlagError(
                f'Unknown partition flag {flag_name}'
            )
        if self.flag_map[flag_name]:
            Command.run(
                [
                    'sgdisk', '-t',
                    ':'.join(
                        [
                            format(partition_id),
                            format(self.flag_map[flag_name])
                        ]
                    ),
                    self.disk_device
                ]
            )
        else:
            log.warning('Flag %s ignored on GPT', flag_name)

    def set_uuid(self, partition_id: int, uuid: str) -> None:
        """
        Set partition UUID (TypeCode)

        :param int partition_id: partition number
        :param string uuid: UUID
        """
        Command.run(
            [
                'sgdisk',
                '--typecode', f'{partition_id}:{uuid}',
                self.disk_device
            ]
        )

    def set_hybrid_mbr(self) -> None:
        """
        Turn partition table into hybrid GPT/MBR table
        """
        partition_ids = []
        partition_number_to_embed = self.partition_count
        if partition_number_to_embed > 3:
            # the max number of partitions to embed is 3
            # for details see man sgdisk
            log.debug(
                'maximum number of GPT hybrid MBR partitions is 3, got %d',
                partition_number_to_embed
            )
            partition_number_to_embed = 3
            log.debug(
                'reduced GPT hybrid MBR partition count to %d',
                partition_number_to_embed
            )
        for number in range(1, partition_number_to_embed + 1):
            if self.partition_map.get(number):
                partition_ids.append(format(self.partition_map[number]))
        Command.run(
            ['sgdisk', '-h', ':'.join(partition_ids), self.disk_device]
        )

    def set_mbr(self) -> None:
        """
        Turn partition table into MBR (msdos table)
        """
        partition_ids = []
        efi_partition_number = None
        for number in range(1, self.partition_id + 1):
            if self.partition_map.get(number):
                partition_info = Command.run(
                    [
                        'sgdisk', '-i={0}'.format(self.partition_map[number]),
                        self.disk_device
                    ]
                )
                if '(EFI System)' in partition_info.output:
                    efi_partition_number = self.partition_map[number]
                partition_ids.append(format(self.partition_map[number]))
        if efi_partition_number:
            # turn former EFI partition into standard linux partition
            self.set_flag(efi_partition_number, 't.linux')
        Command.run(
            ['sgdisk', '-m', ':'.join(partition_ids), self.disk_device]
        )

    def resize_table(self, entries: int = 128) -> None:
        """
        Resize partition table

        :param int entries: number of default entries
        """
        Command.run(
            [
                'sgdisk', '--resize-table', '{0}'.format(entries),
                self.disk_device
            ]
        )
