#! /usr/bin/env python
# -*- coding: utf-8 -*-
#
# graph_tool -- a general graph manipulation python module
#
# Copyright (C) 2006-2025 Tiago de Paula Peixoto <tiago@skewed.de>
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU Lesser General Public License as published by the Free
# Software Foundation; either version 3 of the License, or (at your option) any
# later version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
# details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.

import sys
import os
import datetime

# For OMP performance
os.environ["OMP_WAIT_POLICY"] = "passive"

# If your extensions are in another directory, add it here. If the directory
# is relative to the documentation root, use os.path.abspath to make it
# absolute, like shown here.
sys.path.append(os.path.abspath('.'))

# General configuration
# ---------------------

# Add any Sphinx extension module names here, as strings. They can be extensions
# coming with Sphinx (named 'sphinx.ext.*') or your custom ones.
extensions = [
    'sphinx.ext.autodoc',
    'sphinx.ext.doctest',
    'sphinx.ext.intersphinx',
    'sphinx.ext.mathjax',
    'sphinx.ext.autosummary',
    'numpydoc',
    'sphinx.ext.extlinks',
    'sphinx.ext.viewcode',
    'extlinks_fancy',
    'sphinxext.opengraph',
    'sphinx_copybutton',
    'sphinx_favicon',
    'myst_parser',
    'sphinxcontrib.bibtex',
    'sphinxcontrib.video'
]

myst_enable_extensions = [
    "amsmath",
    "attrs_inline",
    "colon_fence",
    "deflist",
    "dollarmath",
    "fieldlist",
    "html_admonition",
    "html_image",
    "linkify",
    "replacements",
    "smartquotes",
    "strikethrough",
    "substitution",
    "tasklist",
]

bibtex_bibfiles = ['refs.bib']

mathjax_path = "node_modules/mathjax/es5/tex-chtml-full.js"

# Add any paths that contain templates here, relative to this directory.
templates_path = ['_templates']

# The suffix of source filenames.
source_suffix = {'.rst': 'restructuredtext'}

# The encoding of source files.
source_encoding = 'utf-8'

# The master toctree document.
master_doc = 'index'

# General information about the project.
project = u'graph-tool'
year = datetime.datetime.now().year
copyright = f'2006-{year}, Tiago de Paula Peixoto <tiago@skewed.de>'
author = 'Tiago de Paula Peixoto'

# The version info for the project you're documenting, acts as replacement for
# |version| and |release|, also used in various other places throughout the
# built documents.
#
# The short X.Y version.
from graph_tool import __version__ as gt_version
version = gt_version.split()[0]
# The full version, including alpha/beta/rc tags.
release = gt_version.split()[0]

# The language for content autogenerated by Sphinx. Refer to documentation
# for a list of supported languages.
#language = None

# There are two options for replacing |today|: either, you set today to some
# non-false value, then it is used:
#today = ''
# Else, today_fmt is used as the format for a strftime call.
#today_fmt = '%B %d, %Y'

# List of documents that shouldn't be included in the build.
#unused_docs = []

# List of directories, relative to source directory, that shouldn't be searched
# for source files.
exclude_trees = ['.build']

exclude_patterns = ['**/_*.rst', '**/_*.md', 'build/html/_static/*/*.md']

# The reST default role (used for this markup: `text`) to use for all documents.
#default_role = None

# If true, '()' will be appended to :func: etc. cross-reference text.
#add_function_parentheses = True

# If true, the current module name will be prepended to all description
# unit titles (such as .. function::).
#add_module_names = True

# If true, sectionauthor and moduleauthor directives will be shown in the
# output. They are ignored by default.
#show_authors = False

# The name of the Pygments (syntax highlighting) style to use.
pygments_style = 'sphinx'

# doctest

import os
doctest_global_setup = open("pyenv.py").read().replace("___DOCDIR___",
                                                       os.getcwd())
#doctest_global_setup += "os.chdir('%s')\n" % os.getcwd()

# Options for HTML outputs
# -----------------------

# The style sheet to use for HTML and HTML Help pages. A file of that name
# must exist either in Sphinx' static/ path, or in one of the custom paths
# given in html_static_path.
# html_style = 'default.css'

html_theme = "sphinx_book_theme"

# Set up the version switcher.
if "dev" in version:
    switcher_version = "dev"
else:
    switcher_version = f"{version}"

html_theme_options = {
    "gitlab_url": "https://git.skewed.de/count0/graph-tool",
    "icon_links": [
        {
            "name": "Mastodon",
            "url": "https://social.skewed.de/@graph_tool",
            "icon": "fa-brands fa-mastodon",
        },
        {
            "name": "Bluesky",
            "url": "https://bsky.app/profile/graph-tool.skewed.de",
            "icon": "fa-brands fa-bluesky",
        },
        {
            "name": "Discourse",
            "url": "https://forum.skewed.de/c/graph-tool/5",
            "icon": "fa-brands fa-discourse",
        },
        {
            "name": "graph-tool",
            "url": "https://graph-tool.skewed.de",
            "icon": "_static/graph-tool-logo-circle.svg",
            "type": "local",
        },
    ],
    "primary_sidebar_end": ["indices.html"],
    "use_edit_page_button": True,
    "home_page_in_toc": True,
    "show_navbar_depth": 2,
    "show_toc_level": 4,
    'use_sidenotes': False,
    "logo": {
        "image_light": "_static/graph-tool-logo-circle.svg",
        "image_dark": "_static/graph-tool-logo-circle.svg",
        "text": f"""graph-tool documentation<br/>({version})"""
    },
    "switcher": {
        "version_match": switcher_version,
        "json_url": "https://graph-tool.skewed.de/static/docs/versions.json",
    },
    "show_version_warning_banner": True,
}

html_static_path = ["_static"]
html_css_files = ["custom.css", "fonts.css"]

html_context = {
    "gitlab_url": "https://git.skewed.de",
    "gitlab_user": "count0",
    "gitlab_repo": "graph-tool",
    "gitlab_version": "master",
    "doc_path": "/doc"
}

navigation_with_keys = False

html_logo = "_static/graph-tool-logo-circle.svg"

# The name for this set of Sphinx documents.  If None, it defaults to
# "<project> v<release> documentation".
#html_title = None

# A shorter title for the navigation bar.  Default is the same as html_title.
#html_short_title = None

# The name of an image file (relative to this directory) to place at the top
# of the sidebar.
#html_logo = None

# The name of an image file (within the static path) to use as favicon of the
# docs.  This file should be a Windows icon file (.ico) being 16x16 or 32x32
# pixels large.
#html_favicon = "graph-tool-logo-circle.svg"

favicons = [
    {"href": "graph-tool-logo-circle.svg"},
]

# Add any paths that contain custom static files (such as style sheets) here,
# relative to this directory. They are copied after the builtin static files,
# so a file named "default.css" will overwrite the builtin "default.css".
html_static_path = ['_static']

# If not '', a 'Last updated on:' timestamp is inserted at every page bottom,
# using the given strftime format.
html_last_updated_fmt = '%b %d, %Y'

# Custom sidebar templates, maps document names to template names.
#html_sidebars = {}

html_sidebars = {
    "**": ["navbar-logo.html", "version-switcher", "icon-links.html",
           "search-button-field.html", "sbt-sidebar-nav.html"]
}

# Additional templates that should be rendered to pages, maps page names to
# template names.
#html_additional_pages = {}

# If false, no module index is generated.
#html_use_modindex = True

# If false, no index is generated.
#html_use_index = True

# If true, the index is split into individual pages for each letter.
#html_split_index = False

# If true, the reST sources are included in the HTML build as _sources/<name>.
html_copy_source = True

# If true, an OpenSearch description file will be output, and all pages will
# contain a <link> tag referring to it.  The value of this option must be the
# base URL from which the finished HTML is served.
html_use_opensearch = 'http://graph-tool.skewed.de/docs/'

# If nonempty, this is the file name suffix for HTML files (e.g. ".xhtml").
#html_file_suffix = ''

# Output file base name for HTML help builder.
htmlhelp_basename = 'graph-tooldoc'


# Example configuration for intersphinx: refer to the Python standard library.
intersphinx_mapping = {'python': ('https://docs.python.org/3', None),
                       'numpy': ('https://numpy.org/doc/stable/', None),
                       'scipy': ('https://docs.scipy.org/doc/scipy/', None),
                       'matplotlib': ('https://matplotlib.org/stable/', None),
                       'cairo': ('https://pycairo.readthedocs.io/en/latest/', None),
                       'ipython': ('https://ipython.org/ipython-doc/stable/', None),
                       'pandas': ('https://pandas.pydata.org/pandas-docs/stable/', None),
                       'basemap': ('https://matplotlib.org/basemap/stable/', None),
                       'myst': ('https://myst-parser.readthedocs.io/en/latest/', None)
                       }

extlinks_fancy = {'issue': (['https://git.skewed.de/count0/graph-tool/issues/{0}'],
                             ['issue #{0}']),
                  'doi': (['https://dx.doi.org/{0}',
                           'https://sci-hub.ru/{0}',
                           'https://scihub22266oqcxt.onion/{0}'],
                          ['DOI: {0}', "sci-hub", "@tor"]),
                  'arxiv': (['https://arxiv.org/abs/{0}'], ['arXiv: {0}']),
                  'ns': (['https://networks.skewed.de/net/{0}'], ['{0}'])}

autodoc_default_options = {
    'members': False,
    'member-order': 'bysource',
    'undoc-members': False
}

autosummary_gerenerate = True
autosummary_imported_members = True
autosummary_ignore_module_all = True
numpydoc_show_class_members = False
autodoc_docstring_signature = False
autoclass_content = 'both'
imported_members = True

add_module_names = True
toc_object_entries_show_parents = "hide"

ogp_site_url = "https://graph-tool.skewed.de/static/docs/stable/index.html"
ogp_image = "https://graph-tool.skewed.de/static/img/graph-tool-logo.png"
ogp_description_length = 300
ogp_enable_meta_description = True

def linkcode_resolve(domain, info):
    if domain != 'py':
        return None
    if not info['module']:
        return None
    modname = info['module'].replace('.', '/')
    return "https://git.skewed.de/count0/graph-tool/tree/master/src/%s/__init__.py" % modname

nitpicky = True

def skip(app, what, name, obj, would_skip, options):
    if name.startswith("_"):
        return True
    return would_skip

def setup(app):
    app.connect("autodoc-skip-member", skip)

import pybtex.plugin
from pybtex.style.formatting.unsrt import Style as UnsrtStyle
from pybtex.style.template import (
    field, first_of, href, join, names, optional, optional_field, sentence,
    tag, together, words
)

class MyStyle(UnsrtStyle):
    def format_doi(self, e):
        # based on urlbst format.doi
        return join[href[
            join[
                'https://doi.org/',
                field('doi', raw=True)
                ],
            join[
                'doi:',
                field('doi', raw=True)
                ]
            ], ", ", href[
            join[
                'https://sci-hub.ru/',
                field('doi', raw=True)
                ],
            '[sci-hub]'
            ]]

    def format_web_refs(self, e):
        return sentence[
            optional[self.format_url(e)],
            optional[self.format_eprint(e)],
            optional[self.format_pubmed(e)],
            optional[self.format_doi(e)]]

    def format_url(self, e):
        return ""

pybtex.plugin.register_plugin('pybtex.style.formatting', 'mystyle', MyStyle)

bibtex_default_style = 'mystyle'
